/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/


#define _X86_

#include <ddk/ntddk.h>
#include "privatedata.h"


#define SELECTOR_TO_DESCRIPTOR_INDEX(x) (x>>3)


#define ARE_DESCRIPTORS_EQUAL(a, b) \
             (a.wSegmentLength == b.wSegmentLength &&                                         \
             a.bDescriptorFlags == b.bDescriptorFlags &&                                      \
             a.bFlagsAndHighNibbleSegmentLength == b.bFlagsAndHighNibbleSegmentLength &&      \
             a.wBaseAddressLowWord == b.wBaseAddressLowWord &&                                \
             a.bBaseAddressHighWordLowByte == b.bBaseAddressHighWordLowByte &&                \
             a.bBaseAddressHighWordHighByte == b.bBaseAddressHighWordHighByte)


extern GDT_SYSTEM_DESCRIPTOR_ENTRY g_64BitGlobalDescriptorTable[GDT_MAX_ENTRIES];
extern BOOLEAN g_64BitGlobalDescriptorTableEntriesUsed[GDT_MAX_ENTRIES];

/*
 * Internal Functions
 */
unsigned int GDT_ReadDescriptorSegmentBaseAddress(PGDT_SYSTEM_DESCRIPTOR_ENTRY pGdtDescriptorTable, unsigned short usSelectorIndex);
unsigned short GDT_FindExistingDescriptorEntry(PGDT_SYSTEM_DESCRIPTOR_ENTRY pDescriptorEntry);
void GDT_CreateDescriptorEntry(PGDT_SYSTEM_DESCRIPTOR_ENTRY pDescriptorEntry, unsigned short usSelectorIndex);
unsigned short GDT_FindFreeDescriptorEntry(void);
NTSTATUS GDT_CreateCompatibleGDT64(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);



/**********************************************************************
 * 
 *  GDT_ReadDescriptorSegmentBaseAddress
 *
 *    This API will read the segment base address using the
 *    selector index into the GDT.  
 *
 **********************************************************************/
unsigned int GDT_ReadDescriptorSegmentBaseAddress(PGDT_SYSTEM_DESCRIPTOR_ENTRY pGdtDescriptorTable, unsigned short usSelectorIndex)
{
    unsigned int uiBaseAddress = 0;
    unsigned short usTableIndex = (usSelectorIndex >> 3);

    uiBaseAddress  = pGdtDescriptorTable[usTableIndex].wBaseAddressLowWord;
    uiBaseAddress |= (pGdtDescriptorTable[usTableIndex].bBaseAddressHighWordLowByte>>16);
    uiBaseAddress |= (pGdtDescriptorTable[usTableIndex].bBaseAddressHighWordHighByte>>24);

    return uiBaseAddress;
}


/**********************************************************************
 * 
 *  GDT_ReadSelectorInformationFromGDT
 *
 *    This API will get information from the current GDT
 *
 **********************************************************************/
NTSTATUS GDT_ReadSelectorInformationFromGDT(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    PGDT_SYSTEM_DESCRIPTOR_ENTRY pGdtDescriptorTable = NULL;
    NTSTATUS NtStatus = STATUS_UNSUCCESSFUL;
/*    PHYSICAL_ADDRESS PhysicalAddress;*/
    
    pGdtDescriptorTable = (PGDT_SYSTEM_DESCRIPTOR_ENTRY)pOperatingSystemThunkState->OsState32.dt32GdtLocation.usDescriptorTablePhysicalAddress;
/*    pGdtDescriptorTable = (PGDT_SYSTEM_DESCRIPTOR_ENTRY)MmMapIoSpace(PhysicalAddress, pOperatingSystemThunkState->OsState32.dt32GdtLocation.usDescriptorTableSize, MmCached); */

    if(pGdtDescriptorTable)
    {
        pOperatingSystemThunkState->OsDescriptorMapping.uiCSDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiCS);
        pOperatingSystemThunkState->OsDescriptorMapping.pCSDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiCS)];

        pOperatingSystemThunkState->OsDescriptorMapping.uiDSDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiDS);
        pOperatingSystemThunkState->OsDescriptorMapping.pDSDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiDS)];

        pOperatingSystemThunkState->OsDescriptorMapping.uiSSDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiSS);
        pOperatingSystemThunkState->OsDescriptorMapping.pSSDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiSS)];


        pOperatingSystemThunkState->OsDescriptorMapping.uiESDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiES);
        pOperatingSystemThunkState->OsDescriptorMapping.pESDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiES)];

        pOperatingSystemThunkState->OsDescriptorMapping.uiFSDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiFS);
        pOperatingSystemThunkState->OsDescriptorMapping.pFSDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiGS)];

        pOperatingSystemThunkState->OsDescriptorMapping.uiGSDescriptorMapping = GDT_ReadDescriptorSegmentBaseAddress(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.uiGS);
        pOperatingSystemThunkState->OsDescriptorMapping.pGSDescriptor         = &pGdtDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(pOperatingSystemThunkState->OsState32.uiGS)];

/*         MmUnmapIoSpace(pGdtDescriptorTable, pOperatingSystemThunkState->OsState32.dt32GdtLocation.usDescriptorTableSize); */

        NtStatus = STATUS_SUCCESS;
    }

    return NtStatus;
}

/**********************************************************************
 * 
 *  GDT_Setup64BitGlobalDescriptorTable
 *
 *    This API will implement 64 bit global descriptor table
 *
 **********************************************************************/
NTSTATUS GDT_Setup64BitGlobalDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{   
    NTSTATUS NtStatus = STATUS_SUCCESS;

    NtStatus = GDT_ReadSelectorInformationFromGDT(pOperatingSystemThunkState);

    if(NtStatus == STATUS_SUCCESS)
    {
       NtStatus = GDT_CreateCompatibleGDT64(pOperatingSystemThunkState);
    }

    return NtStatus;
}                                 


/**********************************************************************
 * 
 *  GDT_CreateCopyOfDescriptorTable
 *
 *    This API will create a snapshot of the descriptor table
 *
 **********************************************************************/
NTSTATUS GDT_CreateCopyOfDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    PHYSICAL_ADDRESS PhsyicalAddress = { 0xFFFFFFFF };
    NTSTATUS NtStatus = STATUS_UNSUCCESSFUL;

    pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress = (PVOID)MmAllocateContiguousMemory(sizeof(g_64BitGlobalDescriptorTable), PhsyicalAddress);     
    

    if(pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress)
    {
        memcpy(pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress, g_64BitGlobalDescriptorTable, sizeof(g_64BitGlobalDescriptorTable));

        /*
         * Create Descriptor Table Map
         */
        pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.DescriptorTableLoadingAddress.usDescriptorTableSize            = GDT_MAX_ENTRIES*8;
        pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.DescriptorTableLoadingAddress.usDescriptorTablePhysicalAddress = (unsigned int)pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress;

    }

    return NtStatus;
}

/**********************************************************************
 * 
 *  GDT_CreateCompatibleGDT64
 *
 *    This API will setup a compatible GDT
 *
 *      1. Remap the 3 OS descriptors (Data, Code, Stack)
 *      2. Create the Longmode Descriptor
 *      3. Create Descriptor Loadable Mapping
 *
 **********************************************************************/
NTSTATUS GDT_CreateCompatibleGDT64(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;
    
    /*
     * Mirror OS Descriptors
     */
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pCSDescriptor, pOperatingSystemThunkState->OsState32.uiCS);
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pDSDescriptor, pOperatingSystemThunkState->OsState32.uiDS);
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pSSDescriptor, pOperatingSystemThunkState->OsState32.uiSS);
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pESDescriptor, pOperatingSystemThunkState->OsState32.uiES);
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pFSDescriptor, pOperatingSystemThunkState->OsState32.uiFS);
    GDT_CreateDescriptorEntry(pOperatingSystemThunkState->OsDescriptorMapping.pGSDescriptor, pOperatingSystemThunkState->OsState32.uiGS);


    /*
     * Create Special Descriptors
     */
    GDT_CreateLongModeDescriptor(pOperatingSystemThunkState);

    /*
     * Create Descriptor Table Map
     */
    pOperatingSystemThunkState->OsDescriptorMapping.GdtDescriptorTable.usDescriptorTableSize            = GDT_MAX_ENTRIES*8;
    pOperatingSystemThunkState->OsDescriptorMapping.GdtDescriptorTable.usDescriptorTablePhysicalAddress = (unsigned int)&g_64BitGlobalDescriptorTable;

        
    return NtStatus;
}

/**********************************************************************
 * 
 *  GDT_CreateCodeIdentityDescriptor
 *
 *    This API will create a code Identity descriptor
 *
 **********************************************************************/
unsigned short GDT_CreateCodeIdentityDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress)
{
    unsigned short usIdentityDescriptor = 0;
    GDT_SYSTEM_DESCRIPTOR_ENTRY IdentityDescriptor = {0};

    IdentityDescriptor.wSegmentLength                    = 0xFFFF;
    IdentityDescriptor.bDescriptorFlags                  = GDT_PRESENT_BIT | GDT_DPL_0 | GDT_READ_EXECUTE_BIT | GDT_CODE_DESCRIPTOR;
    IdentityDescriptor.bFlagsAndHighNibbleSegmentLength  = 0xF | GDT_DEFAULT_INSTRUCTION_SIZE_32 | GDT_GRANULARITY_PAGE_BIT;

    IdentityDescriptor.wBaseAddressLowWord           =  (uiBaseAddress & 0xFFFF);
    IdentityDescriptor.bBaseAddressHighWordLowByte   =  ((uiBaseAddress >> 16) & 0xFF);
    IdentityDescriptor.bBaseAddressHighWordHighByte  =  ((uiBaseAddress >> 24) & 0xFF);
                                                      
    usIdentityDescriptor = GDT_FindExistingDescriptorEntry(&IdentityDescriptor);

    if(usIdentityDescriptor == 0)
    {
        usIdentityDescriptor = GDT_FindFreeDescriptorEntry();
        GDT_CreateDescriptorEntry(&IdentityDescriptor, usIdentityDescriptor);
    }

    return usIdentityDescriptor;
}

/**********************************************************************
 * 
 *  GDT_CreateDataIdentityDescriptor
 *
 *    This API will create a data Identity descriptor
 *
 **********************************************************************/
unsigned short GDT_CreateDataIdentityDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress)
{
    unsigned short usIdentityDescriptor = 0;
    GDT_SYSTEM_DESCRIPTOR_ENTRY IdentityDescriptor = {0};

    IdentityDescriptor.wSegmentLength                    = 0xFFFF;
    IdentityDescriptor.bDescriptorFlags                  = GDT_PRESENT_BIT | GDT_DPL_0 | GDT_DATA_DESCRIPTOR | GDT_READ_EXECUTE_BIT ;
    IdentityDescriptor.bFlagsAndHighNibbleSegmentLength  = 0xF | GDT_DEFAULT_INSTRUCTION_SIZE_32 | GDT_GRANULARITY_PAGE_BIT;

    IdentityDescriptor.wBaseAddressLowWord           =  (uiBaseAddress & 0xFFFF);
    IdentityDescriptor.bBaseAddressHighWordLowByte   =  ((uiBaseAddress >> 16) & 0xFF);
    IdentityDescriptor.bBaseAddressHighWordHighByte  =  ((uiBaseAddress >> 24) & 0xFF);
                                                      
    usIdentityDescriptor = GDT_FindExistingDescriptorEntry(&IdentityDescriptor);

    if(usIdentityDescriptor == 0)
    {
        usIdentityDescriptor = GDT_FindFreeDescriptorEntry();
        GDT_CreateDescriptorEntry(&IdentityDescriptor, usIdentityDescriptor);
    }

    return usIdentityDescriptor;
}




/**********************************************************************                                                          
 * 
 *  GDT_CreateLongModeDescriptor
 *
 *    This API will create a Longmode descriptor
 *
 **********************************************************************/
unsigned short GDT_CreateLongModeDescriptor(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;
    GDT_SYSTEM_DESCRIPTOR_ENTRY LongModeDescriptor = {0};
    unsigned short usLongModeDescriptor = 0;

    usLongModeDescriptor = GDT_FindFreeDescriptorEntry();

    LongModeDescriptor.bDescriptorFlags                  = GDT_PRESENT_BIT | GDT_DPL_0 | GDT_CODE_DESCRIPTOR;
    LongModeDescriptor.bFlagsAndHighNibbleSegmentLength  = GDT_LONG_MODE_ENABLE;

    GDT_CreateDescriptorEntry( &LongModeDescriptor, usLongModeDescriptor);

    pOperatingSystemThunkState->OsDescriptorMapping.uiLMDescriptorMapping = usLongModeDescriptor;

    return usLongModeDescriptor;
}


/**********************************************************************
 * 
 *  GDT_FindExistingDescriptorEntry
 *
 *    This API will find an existing descriptor in the table
 *    that matches the suggested descriptor.                                      
 *  
 *
 **********************************************************************/
unsigned short GDT_FindExistingDescriptorEntry(PGDT_SYSTEM_DESCRIPTOR_ENTRY pDescriptorEntry)
{
  unsigned short uMatchingSelector = 0; /* 0 = No Match */
  BOOLEAN bFoundMatchingSelector = FALSE;
  unsigned short usSelectorEntries = 1;

  while(!bFoundMatchingSelector && usSelectorEntries < GDT_MAX_ENTRIES)
  {
      if(g_64BitGlobalDescriptorTableEntriesUsed[usSelectorEntries] == TRUE)
      {
          if(ARE_DESCRIPTORS_EQUAL((*pDescriptorEntry), g_64BitGlobalDescriptorTable[usSelectorEntries]))
          {
             uMatchingSelector     = (usSelectorEntries<<3);
             bFoundMatchingSelector = TRUE;
          }
          else
          {
              usSelectorEntries++;
          }
      }
      else
      {
         usSelectorEntries++;
      }
  }

  return uMatchingSelector;
}

/**********************************************************************
 * 
 *  GDT_FindFreeDescriptorEntry
 *
 *    This API will find a free descriptor entry
 *  
 *    This was not implemented to be "thread safe" obviously.
 *    However for our implementation it doesn't need to be and besides
 *    we are running at high IRQL right now, so we'd only run into
 *    trouble if a higher interrupt or multi-processor system did something.
 *
 **********************************************************************/
unsigned short GDT_FindFreeDescriptorEntry(void)
{
  unsigned short usFreeSelector = 0; /* 0 = Error */
  BOOLEAN bFoundFreeSelector = FALSE;
  unsigned short usSelectorEntries = 1;

  while(!bFoundFreeSelector && usSelectorEntries < GDT_MAX_ENTRIES)
  {
      if(g_64BitGlobalDescriptorTableEntriesUsed[usSelectorEntries] == FALSE)
      {
          usFreeSelector     = (usSelectorEntries<<3);
          bFoundFreeSelector = TRUE;
      }
      else
      {
         usSelectorEntries++;
      }
  }

  return usFreeSelector;
}

/**********************************************************************
 * 
 *  GDT_CreateDescriptorEntry
 *
 *    This API will put a descriptor into the descriptor table
 *
 **********************************************************************/
void GDT_CreateDescriptorEntry(PGDT_SYSTEM_DESCRIPTOR_ENTRY pDescriptorEntry, unsigned short usSelectorIndex)
{
    g_64BitGlobalDescriptorTable[SELECTOR_TO_DESCRIPTOR_INDEX(usSelectorIndex)]            = *pDescriptorEntry;
    g_64BitGlobalDescriptorTableEntriesUsed[SELECTOR_TO_DESCRIPTOR_INDEX(usSelectorIndex)] = TRUE;
}

/**********************************************************************
 * 
 *  GDT_Free64BitGlobalDescriptorTable
 *
 *    This API will free 64 bit global descriptor table
 *
 **********************************************************************/
NTSTATUS GDT_Free64BitGlobalDescriptorTable(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    if(pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress)
    {
        MmFreeContiguousMemory(pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress);
        pOperatingSystemThunkState->OsDescriptorMapping.AllocatedDescriptorTable.pOperatingSystemAllocatedAddress = NULL;
    }
            
    return STATUS_SUCCESS;
}


/*
 * Global table that lists free GDT Entries
 *
 */

BOOLEAN g_64BitGlobalDescriptorTableEntriesUsed[GDT_MAX_ENTRIES] =
{
    0
  /* { 0,    0,     0,     0,     0,     0 },  NULL Desctiptor Entry */
};

/* 
 * This is the 64 bit Global Descriptor Table
 *
 *    We will just define "SYSTEM" descriptor
 *    because all the data structures are the same
 *    anyway.
 * Put it at the end of the code so we can align it
 * on a 4k Boundary.
 *
 */
GDT_SYSTEM_DESCRIPTOR_ENTRY g_64BitGlobalDescriptorTable[GDT_MAX_ENTRIES] =
{
    0
  /* { 0,    0,     0,     0,     0,     0 },  NULL Desctiptor Entry */
};




