/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/



                                                                                                                                                  
#ifndef __PAE32_H__
#define __PAE32_H__

/*                                                                                                        +-------------------------+                                                                                           
 *                            [       Virtual Address    ]                                                \     Physical Memory     \                             
 *                                               \                          +-----------------------------+    __________________   \                                                                          
 *                                               \                          \                                 \                 \   \                                                   
 *                                        +------+--> [   Linear Address                           ]          \-----------------\   \                                                                          
 *  32-Bit PAE Paging Overview            \          [   PDP  |    PDE   |       PTE   | Phsyical ]           \                 \   \                                                                    
 *                                        \         31     30  29     21  20        12  11       0            \-----------------\   \                                                                          
 *                                        \            \         \            \                               \                 \   \                                                         
 *                                        \            \         \            \                               \-----------------\<--+                             
 *   [ Selector ]                         \            \         \            \                               \                 \                                 
 *    \     ________                      \            \         \            \  Page Table        +--------> \-----------------\                                 
 *    \    |       |                      \            \         \        +---+    Entries         \          \                 \                                 
 *    +-> |       |-----------------------+            \         \        \       ____________     \          \-----------------\                                 
 *       | GDT   |                           +---------+         \        \      |           |     \          \                 \                                 
 *      |       |                            \                   \        \     |           |      \          \-----------------\                                 
 *     |_______|                             \         +---------+        +--->|           | ------+          \                 \                                 
 *                                           \         \                      |           |                   \-----------------\                                 
 *  The descriptor value                     \         \              +----> |___________|                    \                 \                                 
 *  the paging unless                        \         \              \                                       \-----------------\                                 
 *  compatibility mode                       \         \              \                                       \                 \                                 
 *  is specified.      +---------------------+   +-----+              \         ____________                  \-----------------\                                 
 *                     \                         \                    \        |           |                  \                 \                                 
 *                     \                         \                    \       |           |                   \-----------------\                                 
 *                     \                         \  Page Directory    \      |           |                    \                 \                                 
 *                     \                         \     Tables         \     |           |                     \-----------------\                                 
 *                     \                         \      ____________  \    |___________|                      \                 \                                 
 *                     \                         +---> |           |--+                                       \_________________\                                 
 *                     \                              |           |            ____________                                                                       
 *                     \                             |           |            |           |                                                                       
 *                     \                            |           |            |           |                                                                        
 *                     \   Page Directory     +--->|___________|            |           |                                                                         
 *                     \      Pointer         \                            |           |                                                                          
 *                     \    (4 Entries)       \                           |___________|                                                                           
 *                     \      ____________    \                       
 *                     \     |           |    \                               
 *                     \  . |           |     \       ____________
 *                     +-->|           |------+      |           |
 *                        |___________|             |           | 
 *                     |--                         |           |  
 *                    |                           |           |   
 *                   |                           |___________|    
 *        [         CR3             ]                             
 *
 *
 *
 *     The Virtual Address in EIP is then indexed via the selector into the GDT.  The GDT is then
 *     adds it's base address onto the virtual address to make the linear address.  The linear
 *     address then finally indexes the page tables.
 */




/*
 *  Present Bit
 *
 *  Determines if the page pointed to by this
 *  entry is present in memory
 *
 */ 
#define PAGING_PAE_FLAG_IS_PRESENT       0x01
#define PAGING_PAE_FLAG_IS_PRESENT_MASK  0xFE

/*
 *  Writable Bit
 *
 *  Specifies if the page is writable.
 *
 */
#define PAGING_PAE_FLAG_WRITABLE         0x02
#define PAGING_PAE_FLAG_WRITABLE_MASK    0xFD

/*
 *  User/Supervisor Bit
 *
 *  1 = User Access is allowed
 *
 */
#define PAGING_PAE_FLAG_USER             0x04
#define PAGING_PAE_FLAG_USER_MASK        0xFB


/*
 *  Write Through Caching
 *
 *     Specifies if the page table pointed to
 *     by this entry is write through or write back
 *
 */
#define PAGING_PAE_FLAG_WRITE_THROUGH_CACHE       0x08
#define PAGING_PAE_FLAG_WRITE_THROUGH_CACHE_MASK  0xF7


/*
 *  Page Level Cache Disable
 *
 *     Specifies if the page tables pointed to
 *     by this entry are cachable.
 */
#define PAGING_PAE_FLAG_CACHE_LEVEL_DISABLE       0x10
#define PAGING_PAE_FLAG_CACHE_LEVEL_DISABLE_MASK  0xEF




/*  Global Page Bit 
 *
 *    This allows pages to not be reloaded when
 *    CR3 is reloaded to help in caching.
 *  
 *    The only way to reload a "globally"
 *    cached page is through the INVLPG instruction.
 *
 */
#define PAGING_PAE_FLAG_PAGE_IS_GLOBAL       0x00000001
#define PAGING_PAE_FLAG_PAGE_IS_GLOBAL_MASK  0xFFFFFFFE




/*
 * Attributes
 * 0 = Must be 1 
 * 1 = Reserved.
 * 2 = Reserved.
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 - 8 = Reserved..
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * The other bits are Physical Address of PDP
 */
 

#pragma pack(1)
typedef struct _PAGETABLE_PDP_ENTRY_PAE32
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNotUsed;

} PAGETABLE_PDP_ENTRY_PAE32, *PPAGETABLE_PDP_ENTRY_PAE32;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 - 8 = MBZ - Reserved "MUST BE ZERO" field.
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 */



#pragma pack(1)
typedef struct _PAGETABLE_PDE_ENTRY_PAE32
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNotUsed;

} PAGETABLE_PDE_ENTRY_PAE32, *PPAGETABLE_PDE_ENTRY_PAE32;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = D - Dirty
 * 7 = 0
 * 8 = Global
 * 9 - 11 = AVL - Not used by CPU, available to OS.

 */



#pragma pack(1)
typedef struct _PAGETABLE_PTE_ENTRY_PAE32
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNotUsed;

} PAGETABLE_PTE_ENTRY_PAE32, *PPAGETABLE_PTE_ENTRY_PAE32;



NTSTATUS PhysicalAddressExtensions32_CreatePageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
NTSTATUS PhysicalAddressExtensions32_DestroyPageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);

NTSTATUS PhysicalAddressExtensions32_MapVirtualToPhysical(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress, unsigned int  uiVirtualAddress, unsigned int  uiPhysicalAddress);

#endif






