/***********************************************************************
 * DirectX.CPP
 *  
 *    API for Direct X Demo
 *
 *
 * Toby Opferman Copyright (c) 2003
 *
 ***********************************************************************/
 

                                                                     
#include <windows.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <ddraw.h>
#include <directx.h>


 /***********************************************************************
  * Macros
  ***********************************************************************/

#define COLOR_FIND_MASK(x)   while(!(x & 1)) x>>=1;
#define COLOR_GET_NUMBER_BITS(x, y) while(x) { y++; x>>=1; }
#define COLOR_GOTO_COLOR(x, y, z) if(x > y) { x--; z = FALSE; } else if(x < y) { x++; z = FALSE; } 


 /***********************************************************************
  * Internal Structures
  ***********************************************************************/
  
  typedef struct {
      /* For Quick Access */
      DWORD dwScreenWidth;
      DWORD dwScreenPitch;
      DWORD dwScreenHeight;
      DWORD dwColorDepth;
      DWORD dwNumberRedBits;
      DWORD dwNumberGreenBits;
      DWORD dwNumberBlueBits;
      float fRedMult;
      float fGreenMult;
      float fBlueMult;
      BOOL  bSetFields;
      PUCHAR pVideoBuffer;

      PALETTEENTRY VirtualPalette[256];
      DWORD dwOptimizedPaletteLookUp[256]; // For > 8 Bit Color Modes.  Allows no conversion.

      DDPIXELFORMAT PixelFormat;

  } VIDEOSETUP, *PVIDEOSETUP;

  typedef struct {

      HWND hWnd;
      LPDIRECTDRAW lpDirectDraw;
      LPDIRECTDRAWSURFACE lpDirectDrawSurface;
      LPDIRECTDRAWSURFACE lpDirectBackSurface;
      LPDIRECTDRAWPALETTE lpDirectDrawPalette;
      DDSURFACEDESC ddSurfaceDesc;
      PALETTEENTRY CurPalEntry[256];
      PALETTEENTRY TempPalEntry[256];
      VIDEOSETUP VideoSetup;
      BOOL bWindowedMode;

  } DIRECTX, *PDIRECTX;




 /***********************************************************************
  * DirectX_InitFullScreen
  *  
  *    Initialize Full Screen
  *
  * Parameters
  *     Window  Handle
  * 
  * Return Value
  *     Handle To Cubix
  *
  ***********************************************************************/
HDIRECTX WINAPI DirectX_InitFullScreen(PDIRECTXINIT pDirectXInit)
{
    PDIRECTX pDirectX = NULL;

    if(pDirectX = (PDIRECTX)LocalAlloc(LMEM_ZEROINIT, sizeof(DIRECTX)))
    {
       pDirectX->hWnd = pDirectXInit->hWnd;

       if(SUCCEEDED(DirectDrawCreate(NULL, &pDirectX->lpDirectDraw, NULL)))
       {
           pDirectX->lpDirectDraw->Initialize(NULL);
           
           pDirectX->lpDirectDraw->SetCooperativeLevel(pDirectX->hWnd, DDSCL_FULLSCREEN |
                                                                        DDSCL_ALLOWMODEX |
                                                                        DDSCL_EXCLUSIVE  |
                                                                        DDSCL_ALLOWREBOOT);

           pDirectX->lpDirectDraw->SetDisplayMode(pDirectXInit->dwScreenWidth, pDirectXInit->dwScreenHeight,  pDirectXInit->dwColorDepth);

           pDirectX->VideoSetup.dwColorDepth         = pDirectXInit->dwColorDepth;
           pDirectX->ddSurfaceDesc.dwSize            = sizeof(pDirectX->ddSurfaceDesc);
           pDirectX->ddSurfaceDesc.dwFlags           = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
           pDirectX->ddSurfaceDesc.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE | DDSCAPS_COMPLEX | DDSCAPS_FLIP;
           pDirectX->ddSurfaceDesc.dwBackBufferCount = 1;

           

           if(FAILED(pDirectX->lpDirectDraw->CreateSurface(&pDirectX->ddSurfaceDesc, &pDirectX->lpDirectDrawSurface, NULL)))
           {
               DirectX_UnInit((HDIRECTX)pDirectX);
               pDirectX = NULL;
           }
           else
           {
               pDirectX->ddSurfaceDesc.ddsCaps.dwCaps    = DDSCAPS_BACKBUFFER;
               
               if(FAILED(pDirectX->lpDirectDrawSurface->GetAttachedSurface(&pDirectX->ddSurfaceDesc.ddsCaps, &pDirectX->lpDirectBackSurface)))
               {
                   DirectX_UnInit((PDIRECTX)pDirectX);
                   pDirectX = NULL;
               }
           }

       }
       else
       {
           DirectX_UnInit((HDIRECTX)pDirectX);
           pDirectX = NULL;
       }
    }

    return (HDIRECTX)pDirectX;
}


 /***********************************************************************
  * DirectX_InitWindowMode
  *  
  *    Initialize Windowed Mode
  *
  * Parameters
  *     Window  Handle
  * 
  * Return Value
  *     Handle To Direct X
  *
  ***********************************************************************/
HDIRECTX WINAPI DirectX_InitWindowMode(PDIRECTXINIT pDirectXInit)
{
    PDIRECTX pDirectX = NULL;

    if(pDirectX = (PDIRECTX)LocalAlloc(LMEM_ZEROINIT, sizeof(DIRECTX)))
    {
       pDirectX->hWnd = pDirectXInit->hWnd;

       if(SUCCEEDED(DirectDrawCreate(NULL, &pDirectX->lpDirectDraw, NULL)))
       {
           pDirectX->lpDirectDraw->Initialize(NULL);

           pDirectX->lpDirectDraw->SetCooperativeLevel(pDirectX->hWnd, DDSCL_NORMAL);

           pDirectX->ddSurfaceDesc.dwSize            = sizeof(pDirectX->ddSurfaceDesc);
           pDirectX->ddSurfaceDesc.dwFlags           = DDSD_CAPS;
           pDirectX->ddSurfaceDesc.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE;
           pDirectX->bWindowedMode = TRUE;

           if(FAILED(pDirectX->lpDirectDraw->CreateSurface(&pDirectX->ddSurfaceDesc, &pDirectX->lpDirectDrawSurface, NULL)))
           {
               DirectX_UnInit((HDIRECTX)pDirectX);
               pDirectX = NULL;
           }

       }
       else
       {
           DirectX_UnInit((HDIRECTX)pDirectX);
           pDirectX = NULL;
       }
    }

    return (HDIRECTX)pDirectX;
}

 /***********************************************************************
  * DirectX_UnInit
  *  
  *    UnInitialize Test Graphics
  *
  * Parameters
  *     Handle to Test
  * 
  * Return Value
  *     Nothing
  *
  ***********************************************************************/
void WINAPI DirectX_UnInit(HDIRECTX hDirectX)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;

    if(pDirectX)
    {

        if(pDirectX->lpDirectDrawSurface)
        {
            pDirectX->lpDirectDrawSurface->Release();
            pDirectX->lpDirectDrawSurface = NULL;
        }

        if(pDirectX->lpDirectDraw)
        {
            pDirectX->lpDirectDraw->Release();
            pDirectX->lpDirectDraw = NULL;
        }


        LocalFree(pDirectX);

        pDirectX = NULL;
    }
}

 /***********************************************************************
  * DirectX_BeginPaint
  *  
  *    Get The Video Buffer
  *
  * Parameters
  *     Handle to Test
  * 
  * Return Value
  *     Nothing
  *
  ***********************************************************************/
PUCHAR WINAPI DirectX_BeginPaint(HDIRECTX hDirectX, PUINT pPitch)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;
    UCHAR  *pVideoBuffer = NULL;

    pDirectX->ddSurfaceDesc.dwSize = sizeof(pDirectX->ddSurfaceDesc);

    if(SUCCEEDED(pDirectX->lpDirectBackSurface->Lock(NULL, &pDirectX->ddSurfaceDesc, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL)))
    {
        if(pDirectX->ddSurfaceDesc.dwFlags & DDSD_PITCH)
        {
           pDirectX->VideoSetup.dwScreenPitch = pDirectX->ddSurfaceDesc.lPitch;
        }

        if(!pDirectX->VideoSetup.bSetFields)
        {
            pDirectX->VideoSetup.bSetFields = TRUE;

            if(pDirectX->ddSurfaceDesc.dwFlags & DDSD_HEIGHT)
            {
                pDirectX->VideoSetup.dwScreenHeight = pDirectX->ddSurfaceDesc.dwHeight;
            }

            if(pDirectX->ddSurfaceDesc.dwFlags & DDSD_WIDTH)
            {
                pDirectX->VideoSetup.dwScreenWidth  = pDirectX->ddSurfaceDesc.dwWidth;
            }

            if(pDirectX->ddSurfaceDesc.dwFlags & DDSD_PIXELFORMAT)
            {
                pDirectX->VideoSetup.PixelFormat = pDirectX->ddSurfaceDesc.ddpfPixelFormat;

                if(pDirectX->VideoSetup.PixelFormat.dwFlags & DDPF_RGB && pDirectX->VideoSetup.dwColorDepth > 8)
                {
                    UINT dwFormatWalker;

                    pDirectX->VideoSetup.dwColorDepth = pDirectX->VideoSetup.PixelFormat.dwRGBBitCount;

                    dwFormatWalker = pDirectX->VideoSetup.PixelFormat.dwBBitMask;

                    COLOR_FIND_MASK(dwFormatWalker);
                    COLOR_GET_NUMBER_BITS(dwFormatWalker, pDirectX->VideoSetup.dwNumberBlueBits);

                    dwFormatWalker = pDirectX->VideoSetup.PixelFormat.dwGBitMask;

                    COLOR_FIND_MASK(dwFormatWalker);
                    COLOR_GET_NUMBER_BITS(dwFormatWalker, pDirectX->VideoSetup.dwNumberGreenBits);

                    dwFormatWalker = pDirectX->VideoSetup.PixelFormat.dwRBitMask;

                    COLOR_FIND_MASK(dwFormatWalker);
                    COLOR_GET_NUMBER_BITS(dwFormatWalker, pDirectX->VideoSetup.dwNumberRedBits);

                    pDirectX->VideoSetup.fBlueMult = (float)( ( 1 << pDirectX->VideoSetup.dwNumberBlueBits ) / 256.0);
                    pDirectX->VideoSetup.fRedMult = (float)( ( 1 << pDirectX->VideoSetup.dwNumberRedBits ) / 256.0);
                    pDirectX->VideoSetup.fGreenMult = (float)( ( 1 << pDirectX->VideoSetup.dwNumberGreenBits ) / 256.0);

                }
            }
        }

        if(pPitch)
        {
           *pPitch = pDirectX->VideoSetup.dwScreenPitch;
        }

        pVideoBuffer = (UCHAR *)pDirectX->ddSurfaceDesc.lpSurface;
        pDirectX->VideoSetup.pVideoBuffer = pVideoBuffer;
    }

    return pVideoBuffer;
}



 /***********************************************************************
  * DirectX_EndPaint
  *  
  *    End Painting
  *
  * Parameters
  *     Handle to Test
  * 
  * Return Value
  *     Nothing
  *
  ***********************************************************************/
void WINAPI DirectX_EndPaint(HDIRECTX hDirectX)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;

    pDirectX->lpDirectBackSurface->Unlock(NULL);

    while(FAILED(pDirectX->lpDirectDrawSurface->Flip(NULL, DDFLIP_WAIT))) ;
}


/***********************************************************************
 * DirectX_SetPixelPalette
 *  
 *    Clear Video Buffer
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_SetPixelPalette(HDIRECTX hDirectX, UINT dwScreenX, UINT dwScreenY, UINT dwIndex)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;
    UINT dwColorDepth = pDirectX->VideoSetup.dwColorDepth>>3;

    if(dwScreenX < pDirectX->VideoSetup.dwScreenWidth && dwScreenY < pDirectX->VideoSetup.dwScreenHeight)
    {
        if(dwColorDepth > 1)
        {
            memcpy(&pDirectX->VideoSetup.pVideoBuffer[dwScreenX*dwColorDepth + dwScreenY*pDirectX->VideoSetup.dwScreenPitch], &pDirectX->VideoSetup.dwOptimizedPaletteLookUp[dwIndex], dwColorDepth);
        }
        else
        {
            pDirectX->VideoSetup.pVideoBuffer[dwScreenX + dwScreenY*pDirectX->VideoSetup.dwScreenPitch] = dwIndex;
        }
    }

}

/***********************************************************************
 * DirectX_SetPixelPalette
 *  
 *    Clear Video Buffer
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_SetPixelPaletteBuffer(HDIRECTX hDirectX, PUCHAR pBuffer, UINT dwScreenX, UINT dwScreenY, UINT dwIndex)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;
    UINT dwColorDepth = pDirectX->VideoSetup.dwColorDepth>>3;

    if(dwScreenX < pDirectX->VideoSetup.dwScreenWidth && dwScreenY < pDirectX->VideoSetup.dwScreenHeight)
    {
        if(dwColorDepth > 1)
        {
            memcpy(&pBuffer[dwScreenX*dwColorDepth + dwScreenY*pDirectX->VideoSetup.dwScreenWidth*dwColorDepth], &pDirectX->VideoSetup.dwOptimizedPaletteLookUp[dwIndex], dwColorDepth);
        }
        else
        {
            pBuffer[dwScreenX + dwScreenY*pDirectX->VideoSetup.dwScreenWidth*dwColorDepth] = dwIndex;
        }
    }

}

/***********************************************************************
 * DirectX_SetPixelTrueColor
 *  
 *    Clear Video Buffer
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_SetPixelTrueColor(HDIRECTX hDirectX, UINT dwScreenX, UINT dwScreenY, DWORD dwRgb)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;
    UINT dwColorDepth = pDirectX->VideoSetup.dwColorDepth;

    if(dwScreenX < pDirectX->VideoSetup.dwScreenWidth && dwScreenY < pDirectX->VideoSetup.dwScreenHeight)
    {
       if(dwColorDepth > 8)
       {
          dwColorDepth >>= 3;
          memcpy(&pDirectX->VideoSetup.pVideoBuffer[dwScreenX*dwColorDepth + dwScreenY*pDirectX->VideoSetup.dwScreenPitch], &dwRgb, dwColorDepth);
       }
    }
}

/***********************************************************************
 * DirectX_SetPalette
 *  
 *    
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/

void WINAPI DirectX_SetPalette(HDIRECTX hDirectX, PALETTEENTRY *pPaletteEntries)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;

    if(pDirectX->VideoSetup.dwColorDepth == 8)
    {
        if(!(FAILED(pDirectX->lpDirectDraw->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256, pPaletteEntries, &pDirectX->lpDirectDrawPalette, NULL))))
        {
            pDirectX->lpDirectDrawSurface->SetPalette(pDirectX->lpDirectDrawPalette);
            memcpy(pDirectX->VideoSetup.VirtualPalette, pPaletteEntries, sizeof(PALETTEENTRY)*256);
        }
    }
    else
    {
        DWORD dwIndex;

        memcpy(pDirectX->VideoSetup.VirtualPalette, pPaletteEntries, sizeof(PALETTEENTRY)*256);

        for(dwIndex = 0; dwIndex < 256; dwIndex++)
        {
            pDirectX->VideoSetup.dwOptimizedPaletteLookUp[dwIndex] = DirectX_ConvertPaletteEntryToTrueColor(hDirectX, &pPaletteEntries[dwIndex]);
        }
    }
}

/***********************************************************************
 * DirectX_FadePalette
 *  
 *    
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
BOOL WINAPI DirectX_FadePalette(HDIRECTX hDirectX)
{
   PDIRECTX pDirectX = (PDIRECTX)hDirectX;
   BOOL bFaded = TRUE;
   DWORD dwIndex;

   for(dwIndex = 0; dwIndex < 256; dwIndex++)
   {
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peBlue, 0, bFaded);
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peGreen, 0, bFaded);
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peRed, 0, bFaded);
   }

   if(pDirectX->VideoSetup.dwColorDepth == 8)
   {
       if(!(FAILED(pDirectX->lpDirectDraw->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256, pDirectX->VideoSetup.VirtualPalette, &pDirectX->lpDirectDrawPalette, NULL))))
       {
           pDirectX->lpDirectDrawSurface->SetPalette(pDirectX->lpDirectDrawPalette);
       }
   }
   else
   {
       for(dwIndex = 0; dwIndex < 256; dwIndex++)
       {
           pDirectX->VideoSetup.dwOptimizedPaletteLookUp[dwIndex] = DirectX_ConvertPaletteEntryToTrueColor(hDirectX, &pDirectX->VideoSetup.VirtualPalette[dwIndex]);
       }
   }

   return bFaded;
}


/***********************************************************************
 * DirectX_FadeToPalette
 *  
 *    
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
BOOL WINAPI DirectX_FadeToPalette(HDIRECTX hDirectX, PALETTEENTRY *pPaletteEntries)
{
   PDIRECTX pDirectX = (PDIRECTX)hDirectX;
   BOOL bFaded = TRUE;
   DWORD dwIndex;

   for(dwIndex = 0; dwIndex < 256; dwIndex++)
   {
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peBlue, pPaletteEntries[dwIndex].peBlue, bFaded);
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peGreen, pPaletteEntries[dwIndex].peGreen, bFaded);
      COLOR_GOTO_COLOR(pDirectX->VideoSetup.VirtualPalette[dwIndex].peRed, pPaletteEntries[dwIndex].peRed, bFaded);
   }

   if(pDirectX->VideoSetup.dwColorDepth == 8)
   {
       if(!(FAILED(pDirectX->lpDirectDraw->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256, pDirectX->VideoSetup.VirtualPalette, &pDirectX->lpDirectDrawPalette, NULL))))
       {
           pDirectX->lpDirectDrawSurface->SetPalette(pDirectX->lpDirectDrawPalette);
       }
   }
   else
   {
       for(dwIndex = 0; dwIndex < 256; dwIndex++)
       {
           pDirectX->VideoSetup.dwOptimizedPaletteLookUp[dwIndex] = DirectX_ConvertPaletteEntryToTrueColor(hDirectX, &pDirectX->VideoSetup.VirtualPalette[dwIndex]);
       }
   }

   return bFaded;
}


/***********************************************************************
 * DirectX_ConvertPaletteEntryToTrueColor
 *  
 *    
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
DWORD WINAPI DirectX_ConvertPaletteEntryToTrueColor(HDIRECTX hDirectX, PALETTEENTRY *PaletteEntry)
{
   PDIRECTX pDirectX = (PDIRECTX)hDirectX;
   DWORD dwRgb = 0;
   UCHAR uRed = PaletteEntry->peRed;
   UCHAR uBlue = PaletteEntry->peBlue;
   UCHAR uGreen = PaletteEntry->peGreen;

   if(pDirectX->VideoSetup.dwNumberBlueBits < 8)
     uBlue  = (UCHAR)(uBlue * pDirectX->VideoSetup.fBlueMult);

   if(pDirectX->VideoSetup.dwNumberGreenBits < 8)
     uGreen  = (UCHAR)(uGreen * pDirectX->VideoSetup.fGreenMult);

   if(pDirectX->VideoSetup.dwNumberRedBits < 8)
     uRed  = (UCHAR)(uRed * pDirectX->VideoSetup.fRedMult);
     

   dwRgb = uBlue | (uGreen<<pDirectX->VideoSetup.dwNumberBlueBits) | (uRed << (pDirectX->VideoSetup.dwNumberBlueBits + pDirectX->VideoSetup.dwNumberGreenBits));

   return dwRgb;
}



/***********************************************************************
 * DirectX_GetPalette
 *  
 *    
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_GetPalette(HDIRECTX hDirectX, PALETTEENTRY *pPaletteEntries)
{
    PDIRECTX pDirectX = (PDIRECTX)hDirectX;

    memcpy(pPaletteEntries, pDirectX->VideoSetup.VirtualPalette, sizeof(PALETTEENTRY)*256);
}



/***********************************************************************
 * DirectX_ClearVideoBuffer
 *  
 *    Clear Video Buffer
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_ClearVideoBuffer(HDIRECTX hDirectX)
{
   PDIRECTX pDirectX = (PDIRECTX)hDirectX;
   UINT dwMultiple = pDirectX->VideoSetup.dwColorDepth>>3;
   UINT IndexY;

   for(IndexY = 0; IndexY < pDirectX->VideoSetup.dwScreenHeight; IndexY++)
   {
     memset(&pDirectX->VideoSetup.pVideoBuffer[IndexY*pDirectX->VideoSetup.dwScreenPitch], 0, pDirectX->VideoSetup.dwScreenWidth*dwMultiple);
   }
}


/***********************************************************************
 * DirectX_BlastVideo
 *  
 *    Clear Video Buffer
 *
 * Parameters
 *     Video Buffer
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
void WINAPI DirectX_BlastVideo(HDIRECTX hDirectX, PUCHAR pBuffer)
{
   PDIRECTX pDirectX = (PDIRECTX)hDirectX;
   UINT dwMultiple = pDirectX->VideoSetup.dwColorDepth>>3;
   UINT IndexY;

   for(IndexY = 0; IndexY < pDirectX->VideoSetup.dwScreenHeight; IndexY++)
   {
     memcpy(&pDirectX->VideoSetup.pVideoBuffer[IndexY*pDirectX->VideoSetup.dwScreenPitch], pBuffer + IndexY*pDirectX->VideoSetup.dwScreenWidth*dwMultiple, pDirectX->VideoSetup.dwScreenWidth*dwMultiple);
   }
}



/***********************************************************************
 * DirectX_Debug
 *  
 *    Debug Shit
 *
 *    
 *
 * Parameters
 *     Debug
 *
 * Return Value
 *     Nothing
 *
 ***********************************************************************/
 void DirectX_Debug(char *pszFormatString, ...)
 {
     char DebugString[256];
     va_list vl;

     va_start(vl, pszFormatString);
     vsprintf(DebugString, pszFormatString, vl);
     va_end(vl);

     OutputDebugString(DebugString);
 }


