

#include <windows.h>
#include <stdio.h>
#include <raycaster.h>

//#define STDIO_DBG

#define RC_CONVERT_ANGLE_TO_RADIANS(x) (float)((x)*3.14/180.0)


typedef struct _LEVEL_DATA
{
	PDWORD pdwLevelMap;

	UINT   LevelSizeX;
	UINT   LevelSizeY;

	UINT   CellSizeX;
	UINT   CellSizeY;

	PVOID  pContext;

	PFN_DRAW pfnDrawWallSlice;

} LEVEL_DATA, *PLEVEL_DATA;

typedef struct _LEVEL_LOCATION
{
	float    fIncrementX;
	float    fIncrementY;

	float    fCellLocationX;
	float    fCellLocationY;

	UINT     CellLocationX;
	UINT     CellLocationY;

	UINT     CellIndexX;
	UINT     CellIndexY;

	UINT     LastIntersectionX;
	UINT     LastIntersectionY;

	float     fLastIncrementX;
	float     fLastIncrementY;

	BOOL   XIsBlocked;
	BOOL   YIsBlocked;

	float     CellIntersectLocationX;
	float    CellIntersectLocationY;

	float    fRadianDirection;
	int      DirectionAngle;

} LEVEL_LOCATION, *PLEVEL_LOCATION;


typedef struct _RAYCASTER
{
    LEVEL_DATA LevelData;
    LEVEL_LOCATION LevelLocation;


} RAYCASTER, *PRAYCASTER;


/******************************************************* 
 * Internal Prototypes
 *******************************************************/
DWORD RayCaster_MoveBackwards(PRAYCASTER pRayCaster, PLEVEL_LOCATION pLevelLocation);
DWORD RayCaster_MoveForwards(PRAYCASTER pRayCaster, PLEVEL_LOCATION pLevelLocation);

/**********************************************************************
 *
 *  RayCaster_Init
 *
 *
 *
 **********************************************************************/
HRAYCAST RayCaster_Init(PRAYCAST_INIT pRayCastInit)
{
	PRAYCASTER pRayCaster = NULL;
	DWORD MapSize;

	pRayCaster = (PRAYCASTER)LocalAlloc(LMEM_ZEROINIT, sizeof(RAYCASTER));

    if(pRayCaster)
	{
		pRayCaster->LevelData.LevelSizeX = pRayCastInit->LevelSizeX;
		pRayCaster->LevelData.LevelSizeY = pRayCastInit->LevelSizeY;
		pRayCaster->LevelData.CellSizeX  = pRayCastInit->CellSizeX;
		pRayCaster->LevelData.CellSizeY  = pRayCastInit->CellSizeY;
		pRayCaster->LevelData.pfnDrawWallSlice = pRayCastInit->pfnDrawWallSlice;
		pRayCaster->LevelData.pContext   = pRayCastInit->pContext;

		pRayCaster->LevelLocation.CellLocationX = pRayCastInit->CellLocationX;
		pRayCaster->LevelLocation.CellLocationY = pRayCastInit->CellLocationY;
		pRayCaster->LevelLocation.fCellLocationX = (float)pRayCastInit->CellLocationX;
		pRayCaster->LevelLocation.fCellLocationY = (float)pRayCastInit->CellLocationY;
		pRayCaster->LevelLocation.CellIndexX = pRayCastInit->CellIndexX;
		pRayCaster->LevelLocation.CellIndexY = pRayCastInit->CellIndexY;

	    pRayCaster->LevelLocation.fRadianDirection = RC_CONVERT_ANGLE_TO_RADIANS(pRayCastInit->DirectionAngle);
		pRayCaster->LevelLocation.DirectionAngle   = pRayCastInit->DirectionAngle;
	    pRayCaster->LevelLocation.fIncrementX      = (float)cos(pRayCaster->LevelLocation.fRadianDirection);
	    pRayCaster->LevelLocation.fIncrementY      = (float)sin(pRayCaster->LevelLocation.fRadianDirection);

		MapSize = sizeof(DWORD)*pRayCastInit->LevelSizeX*pRayCastInit->LevelSizeY;
		
		pRayCaster->LevelData.pdwLevelMap = (PDWORD)LocalAlloc(LMEM_ZEROINIT, MapSize);
		
		if(pRayCaster->LevelData.pdwLevelMap)
		{
			memcpy(pRayCaster->LevelData.pdwLevelMap, pRayCastInit->pdwLevelMap, MapSize);
		}	
		else
		{
			LocalFree(pRayCaster);
			pRayCaster = NULL;
		}
	}

	return (HRAYCAST)pRayCaster;
}



/**********************************************************************
 *
 *  RayCaster_MoveBackwards
 *
 *
 *
 **********************************************************************/
DWORD RayCaster_MoveBackwards(PRAYCASTER pRayCaster, PLEVEL_LOCATION pLevelLocation)
{
	DWORD MoveBlocked = FALSE;
	float fNewX, fNewY;
	BOOL LocationXIsNextCell;
	BOOL LocationYIsNextCell;
	UINT NewCellX;
	UINT NewCellY;
	int iNewX, iNewY;

	fNewX = pLevelLocation->fCellLocationX - pLevelLocation->fIncrementX;
	fNewY = pLevelLocation->fCellLocationY - pLevelLocation->fIncrementY;
	iNewX = (int)fNewX;
	iNewY = (int)fNewY;

#ifdef STDIO_DBG
	printf("RayCaster_MoveForwards(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fCellLocationX, pLevelLocation->fCellLocationY, iNewX, iNewY);
	printf("     Increment(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fIncrementX, pLevelLocation->fIncrementY, pLevelLocation->CellLocationX, pLevelLocation->CellLocationY);
#endif

	LocationXIsNextCell = (fNewX < 0 || fNewX > (pRayCaster->LevelData.CellSizeX-1)) ? TRUE : FALSE;
	LocationYIsNextCell = (fNewY < 0 || fNewY > (pRayCaster->LevelData.CellSizeY-1)) ? TRUE : FALSE;

	if(LocationXIsNextCell || LocationYIsNextCell)
	{
		NewCellX = pLevelLocation->CellIndexX;

		if(LocationXIsNextCell)
		{
			if(fNewX < 0)
			{
				fNewX = (pRayCaster->LevelData.CellSizeX - 1) + fNewX;
				iNewX = (int)fNewX;
				NewCellX--;
			}
			else
			{
				fNewX = fNewX - (pRayCaster->LevelData.CellSizeX - 1);
				iNewX = (int)fNewX;
				NewCellX++;
			}
		}
		else
		{
			pLevelLocation->fCellLocationX = fNewX;
			pLevelLocation->CellLocationX = iNewX;
		}

		NewCellY = pLevelLocation->CellIndexY;

		if(LocationYIsNextCell)
		{
			if(fNewY < 0)
			{
				fNewY = (pRayCaster->LevelData.CellSizeY - 1) + fNewY;
				iNewY = (int)fNewY;
				NewCellY--;
			}
			else
			{
				fNewY = fNewY - (pRayCaster->LevelData.CellSizeY - 1);
				iNewY = (int)fNewY;
				NewCellY++;
			}
		}
		else
		{
			pLevelLocation->fCellLocationY = fNewY;
			pLevelLocation->CellLocationY = iNewY;
		}

		MoveBlocked = pRayCaster->LevelData.pdwLevelMap[NewCellX + (NewCellY*pRayCaster->LevelData.LevelSizeX)];

		if(MoveBlocked == 0)
		{
			pLevelLocation->fCellLocationX = fNewX;
			pLevelLocation->fCellLocationY = fNewY;
			pLevelLocation->CellLocationX = iNewX;
			pLevelLocation->CellLocationY = iNewY;

			pLevelLocation->CellIndexX = NewCellX;
			pLevelLocation->CellIndexY = NewCellY;
		}
		else
		{
			pLevelLocation->LastIntersectionX = NewCellX;
	        pLevelLocation->LastIntersectionY = NewCellY;
		}

	}
	else
	{
		pLevelLocation->fCellLocationX = fNewX;
		pLevelLocation->fCellLocationY = fNewY;
		pLevelLocation->CellLocationX = iNewX;
		pLevelLocation->CellLocationY = iNewY;
	}

#ifdef STDIO_DBG
	printf("   Final(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fCellLocationX, pLevelLocation->fCellLocationY, pLevelLocation->CellLocationX, pLevelLocation->CellLocationY);
#endif


	return MoveBlocked;
}



/**********************************************************************
 *
 *  RayCaster_MoveForwards
 *
 *
 *
 **********************************************************************/
DWORD RayCaster_MoveForwards(PRAYCASTER pRayCaster, PLEVEL_LOCATION pLevelLocation)
{
	DWORD MoveBlocked = 0;
	float fNewX, fNewY;
	BOOL LocationXIsNextCell;
	BOOL LocationYIsNextCell;
	UINT NewCellX;
	UINT NewCellY;
	int iNewX, iNewY;
	float oLocationX, oLocationY;
	float fEndX, fEndY;
	float fRollOverX, fRollOverY;

	pLevelLocation->LastIntersectionX = NO_INTERSECTION;
	pLevelLocation->LastIntersectionY = NO_INTERSECTION;

	oLocationX = pLevelLocation->fCellLocationX;
	oLocationY = pLevelLocation->fCellLocationY;

	fNewX = pLevelLocation->fCellLocationX + pLevelLocation->fIncrementX;
	fNewY = pLevelLocation->fCellLocationY + pLevelLocation->fIncrementY;
	iNewX = (int)fNewX;
	iNewY = (int)fNewY;

#ifdef STDIO_DBG
	printf("RayCaster_MoveForwards(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fCellLocationX, pLevelLocation->fCellLocationY, iNewX, iNewY);
	printf("     Increment(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fIncrementX, pLevelLocation->fIncrementY, pLevelLocation->CellLocationX, pLevelLocation->CellLocationY);
#endif

	LocationXIsNextCell = (fNewX < 0 || fNewX > (pRayCaster->LevelData.CellSizeX-1)) ? TRUE : FALSE;
	LocationYIsNextCell = (fNewY < 0 || fNewY > (pRayCaster->LevelData.CellSizeY-1)) ? TRUE : FALSE;

	if(LocationXIsNextCell || LocationYIsNextCell)
	{

		NewCellX = pLevelLocation->CellIndexX;

		if(LocationXIsNextCell)
		{
			if(fNewX < 0)
			{
				fNewX = (pRayCaster->LevelData.CellSizeX - 1) + fNewX;
				iNewX = (int)fNewX;
				NewCellX--;
				fEndX = pLevelLocation->fCellLocationX;
				fRollOverX = (pRayCaster->LevelData.CellSizeX - 1) - fNewX;
			}
			else
			{
				fNewX = fNewX - (pRayCaster->LevelData.CellSizeX - 1);
				iNewX = (int)fNewX;
				NewCellX++;
				fRollOverX = fNewX;
				fEndX = (pRayCaster->LevelData.CellSizeX - 1) - pLevelLocation->fCellLocationX;
			}
		}
		else
		{
			pLevelLocation->fCellLocationX = fNewX;
			pLevelLocation->CellLocationX = iNewX;
			pLevelLocation->fLastIncrementX = pLevelLocation->fCellLocationX - oLocationX;
		}

		NewCellY = pLevelLocation->CellIndexY;

		if(LocationYIsNextCell)
		{
			if(fNewY < 0)
			{
				fNewY = (pRayCaster->LevelData.CellSizeY - 1) + fNewY;
				iNewY = (int)fNewY;
				NewCellY--;
				fEndY = pLevelLocation->fCellLocationY;
				fRollOverY = (pRayCaster->LevelData.CellSizeY - 1) - fNewY;
			}
			else
			{
				fNewY = fNewY - (pRayCaster->LevelData.CellSizeY - 1);
				iNewY = (int)fNewY;
				NewCellY++;
				fRollOverY = fNewY;
				fEndY = (pRayCaster->LevelData.CellSizeY - 1) - pLevelLocation->fCellLocationY;
			}
		}
		else
		{
			pLevelLocation->fCellLocationY = fNewY;
			pLevelLocation->CellLocationY = iNewY;
			pLevelLocation->fLastIncrementY = pLevelLocation->fCellLocationY - oLocationY;
		}

		MoveBlocked = pRayCaster->LevelData.pdwLevelMap[NewCellX + (NewCellY*pRayCaster->LevelData.LevelSizeX)];

		if(MoveBlocked == 0)
		{
			pLevelLocation->fCellLocationX = fNewX;
			pLevelLocation->fCellLocationY = fNewY;
			pLevelLocation->CellLocationX = iNewX;
			pLevelLocation->CellLocationY = iNewY;

			pLevelLocation->CellIndexX = NewCellX;
			pLevelLocation->CellIndexY = NewCellY;
			if(LocationXIsNextCell)
			{
				pLevelLocation->fLastIncrementX = fEndX + fRollOverX;
			}

			if(LocationYIsNextCell)
			{
				pLevelLocation->fLastIncrementY = fEndY + fRollOverY;
			}

		}
		else
		{


			pLevelLocation->LastIntersectionX = NewCellX;
	        pLevelLocation->LastIntersectionY = NewCellY;
			pLevelLocation->XIsBlocked = LocationXIsNextCell;
			pLevelLocation->YIsBlocked = LocationYIsNextCell;

			if(LocationXIsNextCell)
			{
				pLevelLocation->fLastIncrementX   = fEndX;
			}

			if(LocationYIsNextCell)
			{
				pLevelLocation->fLastIncrementY = fEndY;
			}
		}



	}
	else
	{
		pLevelLocation->fCellLocationX = fNewX;
		pLevelLocation->fCellLocationY = fNewY;
		pLevelLocation->CellLocationX = iNewX;
		pLevelLocation->CellLocationY = iNewY;
		pLevelLocation->fLastIncrementX = pLevelLocation->fCellLocationX - oLocationX;
		pLevelLocation->fLastIncrementY = pLevelLocation->fCellLocationY - oLocationY;
	}



#ifdef STDIO_DBG
	printf("   Final(%1.2f, %1.2f) (%i, %i)\n", pLevelLocation->fCellLocationX, pLevelLocation->fCellLocationY, pLevelLocation->CellLocationX, pLevelLocation->CellLocationY);
#endif


	return MoveBlocked;
}


/**********************************************************************
 *
 *  RayCaster_Turn
 *
 *
 *
 **********************************************************************/
void RayCaster_Turn(HRAYCAST hRayCast, int TurnAngleMod)
{
	PRAYCASTER pRayCaster = (PRAYCASTER)hRayCast;

	if(TurnAngleMod)
	{
		pRayCaster->LevelLocation.DirectionAngle += TurnAngleMod;

		while(pRayCaster->LevelLocation.DirectionAngle >= 360)
		{
			pRayCaster->LevelLocation.DirectionAngle -= 360;
		}

		while(pRayCaster->LevelLocation.DirectionAngle < 0)
		{
			pRayCaster->LevelLocation.DirectionAngle += 360;
		}

		pRayCaster->LevelLocation.fRadianDirection = RC_CONVERT_ANGLE_TO_RADIANS(pRayCaster->LevelLocation.DirectionAngle);
	}

	pRayCaster->LevelLocation.fIncrementX      = (float)cos(pRayCaster->LevelLocation.fRadianDirection);
	pRayCaster->LevelLocation.fIncrementY      = (float)sin(pRayCaster->LevelLocation.fRadianDirection);
}


/**********************************************************************
 *
 *  RayCaster_Move
 *
 *
 *
 **********************************************************************/
DWORD RayCaster_Move(HRAYCAST hRayCast, int NumberSteps)
{
	PRAYCASTER pRayCaster = (PRAYCASTER)hRayCast;
	DWORD WallItem = 0;
		
	while(NumberSteps < 0 && WallItem == 0)
	{
     	WallItem = RayCaster_MoveBackwards(pRayCaster, &pRayCaster->LevelLocation);
		NumberSteps++;
	}

	while(NumberSteps > 0 && WallItem == 0)
	{
		WallItem = RayCaster_MoveForwards(pRayCaster, &pRayCaster->LevelLocation);
		NumberSteps--;
	}

    return  WallItem;
}


/**********************************************************************
 *
 *  RayCaster_GetLocation
 *
 *
 *
 **********************************************************************/
void RayCaster_GetLocation(HRAYCAST hRayCast, PMAP_LOCATION pMapLocation)
{
	PRAYCASTER pRayCaster = (PRAYCASTER)hRayCast;

	pMapLocation->CellLocationX = pRayCaster->LevelLocation.CellLocationX;
	pMapLocation->CellLocationY = pRayCaster->LevelLocation.CellLocationY;
	pMapLocation->CellIndexX = pRayCaster->LevelLocation.CellIndexX;
	pMapLocation->CellIndexY = pRayCaster->LevelLocation.CellIndexY;
	pMapLocation->DirectionAngle = pRayCaster->LevelLocation.DirectionAngle;

}


/**********************************************************************
 *
 *  RayCaster_Cast
 *
 *
 *
 **********************************************************************/
void RayCaster_Cast(HRAYCAST hRayCast, UINT Width, UINT CellSize, UINT ViewAngle)
{
	PRAYCASTER pRayCaster = (PRAYCASTER)hRayCast;
	LEVEL_LOCATION LevelWalker;
	float fRadians;
	float fRadiansIncrement;
	float fDistanceX;
	float fDistanceY;
	DWORD WallCollision;
	int RayIndex;
	int Distance;
	DRAW_CONTEXT DrawContext;
	float fviewradians;
	
	fRadians          = RC_CONVERT_ANGLE_TO_RADIANS(ViewAngle);
	fviewradians      = -1*fRadians/2;
	fRadians          = (float)(pRayCaster->LevelLocation.fRadianDirection - ((fRadians)/2.0));
	fRadiansIncrement = RC_CONVERT_ANGLE_TO_RADIANS((float)ViewAngle/(float)Width);


	DrawContext.pContext = pRayCaster->LevelData.pContext;

	for(RayIndex = (int)Width - 1; RayIndex >= 0; fRadians += fRadiansIncrement, RayIndex--, fviewradians += fRadiansIncrement)
	{
		fDistanceX = 0;
		fDistanceY = 0;

		memcpy(&LevelWalker, &pRayCaster->LevelLocation, sizeof(LEVEL_LOCATION));

		DrawContext.VerticleLine = RayIndex;

		LevelWalker.fIncrementY = (float)sin(fRadians);
		LevelWalker.fIncrementX = (float)cos(fRadians);
		LevelWalker.fRadianDirection = fRadians;

		WallCollision = FALSE;

		while(WallCollision == 0)
		{
			WallCollision = RayCaster_MoveForwards(pRayCaster, &LevelWalker);
			fDistanceX += LevelWalker.fLastIncrementX;
			fDistanceY += LevelWalker.fLastIncrementY;
		}

		DrawContext.ImageNumber = WallCollision;
		Distance = (int)(sqrt(fDistanceX*fDistanceX + fDistanceY*fDistanceY)*cos(fviewradians));
		
		DrawContext.Distance    = (UINT)(Distance < 0 ? (-1*Distance) : Distance);

		DrawContext.XIsBlocked =	LevelWalker.XIsBlocked;
	    DrawContext.YIsBlocked =	LevelWalker.YIsBlocked;

		fDistanceX += pRayCaster->LevelLocation.fCellLocationX;
		fDistanceY += pRayCaster->LevelLocation.fCellLocationY;
		
		if(fDistanceX < 0)
		{
			fDistanceX = -1*fDistanceX;
		}

    	if(fDistanceY < 0)
		{
			fDistanceY = -1*fDistanceY;
		}

		DrawContext.CellLocationX =	((UINT)fDistanceX) % CellSize;
		DrawContext.CellLocationY = ((UINT)fDistanceY) % CellSize;

		DrawContext.CellX =	LevelWalker.LastIntersectionX;
	    DrawContext.CellY =	LevelWalker.LastIntersectionY;

		pRayCaster->LevelData.pfnDrawWallSlice(hRayCast, &DrawContext);
	}
}


